<?php

namespace App\Http\Controllers\Backend\InventorySetup;

use App\Models\Goods;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Services\InventorySetup\GoodsService;
use App\Services\InventorySetup\GenreService;
use App\Services\InventorySetup\UnitService;
use App\Transformers\GoodsTransformer;
use App\Transformers\Transformers;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;

class GoodsController extends Controller
{
    /**
     * @var GenreService
     */
    private $genreService;
    /**
     * @var UnitService
     */
    private $unitService;
    /**
     * @var GoodsService
     */
    private $systemService;
    /**
     * @var GoodsTransformer
     */
    private $systemTransformer;

    /**
     * CategoryController constructor.
     * @param GoodsService $systemService
     * @param GoodsService $systemTransformer
     */
    public function __construct(UnitService $unitService, GenreService $genreService, GoodsService $goodsService, Transformers $goodsTransformer)
    {
        $this->genreService = $genreService;
        $this->unitService = $unitService;
        $this->systemService = $goodsService;
        $this->systemTransformer = $goodsTransformer;
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request) {
        $title = 'Goods List';
        return view('backend.pages.inventories.goods.index', get_defined_vars());
    }


    public function dataProcessingGoods(Request $request) {
        $json_data = $this->systemService->getList($request);
        return json_encode($this->systemTransformer->dataTable($json_data));
    }



    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create() {
        $title = 'Add New Goods';
        $genres = $this->genreService->getAllList();
        $units = $this->unitService->getAllList();
        $goodsLastData = Goods::latest('id')->first();
        if ($goodsLastData) :
            $goodsData = $goodsLastData->id + 1;
        else :
            $goodsData = 1;
        endif;
        $goodsCode = 'GD' . str_pad($goodsData, 5, "0", STR_PAD_LEFT);
        return view('backend.pages.inventories.goods.create', get_defined_vars());
    }
    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request) {
        try {
            $this->validate($request, $this->systemService->storeValidation($request));
        } catch (ValidationException $e) {
            session()->flash('error', 'Validation error !!');
            return redirect()->back()->withErrors($e->errors())->withInput();
        }
        $this->systemService->store($request);
        session()->flash('success', 'Data successfully save!!');
        return redirect()->route('inventorySetup.goods.index');
    }
    /**
     *
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($id) {
        if (!is_numeric($id)) {
            session()->flash('error', 'Edit id must be numeric!!');
            return redirect()->back();
        }
        $editInfo =   $this->systemService->details($id);
        if (!$editInfo) {
            session()->flash('error', 'Edit info is invalid!!');
            return redirect()->back();
        }
        $title = 'Edit Goods';
        $genres = $this->genreService->getAllList();
        // $units = $this->unitService->getAllList();
        return view('backend.pages.inventories.goods.edit', get_defined_vars());
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id) {
        if (!is_numeric($id)) {
            session()->flash('error', 'Edit id must be numeric!!');
            return redirect()->back();
        }
        $editInfo = $this->systemService->details($id);
        if (!$editInfo) {
            session()->flash('error', 'Edit info is invalid!!');
            return redirect()->back();
        }

        try {
            $this->validate($request, $this->systemService->updateValidation($request, $id));
        } catch (ValidationException $e) {
            session()->flash('error', 'Validation error !!');
            return redirect()->back()->withErrors($e->errors())->withInput();
        }
        $this->systemService->update($request, $id);
        session()->flash('success', 'Data successfully updated!!');
        return redirect()->route('inventorySetup.goods.index');
    }

    /**
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function statusUpdate($id, $status) {
        if (!is_numeric($id)) {
            return response()->json($this->systemTransformer->invalidId($id), 200);
        }
        $detailsInfo =   $this->systemService->details($id);
        if (!$detailsInfo) {
            return response()->json($this->systemTransformer->notFound($detailsInfo), 200);
        }
        $statusInfo =  $this->systemService->statusUpdate($id, $status);
        if ($statusInfo) {
            return response()->json($this->systemTransformer->statusUpdate($statusInfo), 200);
        }
    }


    /**
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function destroy($id) {
        if (!is_numeric($id)) {
            return response()->json($this->systemTransformer->invalidId($id), 200);
        }
        $detailsInfo =   $this->systemService->details($id);
        if (!$detailsInfo) {
            return response()->json($this->systemTransformer->notFound($detailsInfo), 200);
        }
        $deleteInfo =  $this->systemService->destroy($id);
        if ($deleteInfo) {
            return response()->json($this->systemTransformer->delete($deleteInfo), 200);
        }
    }

    /**
     * Delete the thumbnail of a product.
     *
     * @param int $id The ID of the product whose thumbnail is to be deleted.
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteThumbnail($id) {
        if (!is_numeric($id)) {
            session()->flash('error', 'ID must be numeric!!');
            return redirect()->back();
        }

        $deleteThumbnail = $this->systemService->deleteThumbnail($id);

        if ($deleteThumbnail) {
            session()->flash('success', 'Thumbnail deleted successfully!');
        } else {
            session()->flash('error', 'Failed to delete thumbnail!');
        }

        return redirect()->back();
    }
}
